using System.Net;
#if NUNIT
#region Directives
//-----------------------***
using System;
using System.Collections;
using System.Data;
using System.Text;
using NUnit.Framework;
using gov.va.med.vbecs.UnitTest.VAL;
using gov.va.med.vbecs.UnitTests;

using TABLES = gov.va.med.vbecs.Common.VbecsTables;
using SPROCS = gov.va.med.vbecs.Common.VbecsStoredProcs;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;


using BOL = gov.va.med.vbecs.BOL;
using COMMON = gov.va.med.vbecs.Common;
using DAL = gov.va.med.vbecs.DAL;
using VAL = gov.va.med.vbecs.VAL;
using gov.va.med.vbecs.VAL.UnitTests;
//-----------------------***
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules;

#endregion
namespace gov.va.med.vbecs.BOL.UnitTests
{

	
	

	[TestFixture]
	public class BOL_VistALinkConfig : BaseTest 
	{
		private const int InvalidVistaPortNumber = -1;

		[SetUp]
		protected void SetUp()
		{
			this.BaseSetUp();

			DataCreator.ResetVistaLinkControlTable();
		}

		[TearDown]
		protected void TearDown()
		{
			DataCreator.ResetVistaLinkControlTable();
		}

		private VistALinkConfig GetNonNullVerifiedConfig()
		{
			VistALinkConfig _cfg = VistALinkConfig.GetCurrent();
			
			Assert.IsNotNull( _cfg ); 

			return _cfg;
		}

		[Test]
		public void GetCurrent_Pass_1()
		{
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();

			VerifyObject( _cfg, false, false, 0, null );
		}

		[Test]
		public void GetCurrent_Pass_2()
		{
            UnitTestUtil.RunSQL("DELETE FROM " + TABLES.VistaLinkControl.TableName); 
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();

			VerifyObject( _cfg, true, false, 3, null );
		}

		[Test]
		[Ignore( StandardIgnoreReasons.ParameterlessMethod )]
		public void GetCurrent_Fail() {}

		[Test]
		public void LoadDataRowFromThis_Pass()
		{
			VistALinkConfig _cfg = VistALinkConfig.GetCurrent();
			DataRow _dr = DAL.VistALinkConfig.GetVistaLinkControlEmptyDataTable().NewRow();

			_cfg.LoadDataRowFromThis( _dr );

			Assert.AreEqual( _cfg.VistaIpAddress.ToString(), Convert.ToString( _dr[ TABLES.VistaLinkControl.VistaIpAddress ] ).Trim() );
			Assert.AreEqual( _cfg.VistaPortNumber, Convert.ToInt32( _dr[ TABLES.VistaLinkControl.VistaPortNumber] ) );

			VerifyObject( _cfg, false, false, 0, null );
		}

		[Test]
		[ExpectedException(typeof(ArgumentNullException))]
		public void LoadDataRowFromThis_Fail_1()
		{
			VistALinkConfig.GetCurrent().LoadDataRowFromThis( null );
		}

		[Test]
		[ExpectedException(typeof(BusinessObjectException))]
		public void LoadDataRowFromThis_Fail_2()
		{
            UnitTestUtil.RunSQL("DELETE FROM " + TABLES.VistaLinkControl.TableName); 
			VistALinkConfig.GetCurrent().LoadDataRowFromThis( DAL.VistALinkConfig.GetVistaLinkControlEmptyDataTable().NewRow() );
		}

		[Test]
		public void GetServerConnectionInfo_Pass()
		{
			Assert.IsNotNull( VistALinkConfig.GetCurrent().GetServerConnectionInfo() );
		}

		[Test]
		[Ignore( StandardIgnoreReasons.ParameterlessMethod )]
		public void GetServerConnectionInfo_Fail() {}

		private void VerifyObject( VistALinkConfig configToVerify, bool isNew, bool isDirty, int expectedNumberOfBrokenRules, string expectedBrokenRule )
		{
			if( configToVerify == null )
				Assert.Fail( String.Format( "Null reference was passed to object verification method '{0}'", System.Reflection.MethodBase.GetCurrentMethod().Name ) );

			Assert.AreEqual( isNew, configToVerify.IsNew );
			Assert.AreEqual( isDirty, configToVerify.IsDirty );
			Assert.AreEqual( expectedNumberOfBrokenRules, configToVerify.BrokenRulesCount );
			
			if( expectedBrokenRule != null )
				Assert.IsTrue( configToVerify.IsRuleBroken( expectedBrokenRule ) );
		}

		[Test]
		public void VistaIpAddress_Pass_()
		{
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();

			_cfg.VistaIpAddress = VistALinkUnitTestHarness.TestData.VistaIpAddress;
			
			VerifyObject( _cfg, false, true, 0, null );
			Assert.IsTrue( _cfg.VistaIpAddress.Equals( VistALinkUnitTestHarness.TestData.VistaIpAddress ) );
		}

		[Test]
		public void VistaIpAddress_Fail_1()
		{
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();
			_cfg.VistaDomainName = null;
			_cfg.VistaIpAddress = null;

			VerifyObject( _cfg, false, true, 1, BROKEN.VistaLinkConfig.VistaServerIpAddressAndDomainNameNotSpecified );
			Assert.IsNull( _cfg.VistaIpAddress );
		}


		[Test]
		public void VistaDomainName_Pass_()
		{
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();

			_cfg.VistaDomainName = VistALinkUnitTestHarness.TestData.DomainName;
			
			VerifyObject( _cfg, false, true, 0, null );
			Assert.IsTrue( _cfg.VistaDomainName.Equals( VistALinkUnitTestHarness.TestData.DomainName ) );
		}

		[Test]
		public void VistaDomainName_Fail_1()
		{
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();
			_cfg.VistaIpAddress = null;
			_cfg.VistaDomainName = null;

			VerifyObject( _cfg, false, true, 1, BROKEN.VistaLinkConfig.VistaServerIpAddressAndDomainNameNotSpecified );
			Assert.IsNull( _cfg.VistaDomainName );
		}



		[Test]
		public void VistaPortNumber_Pass()
		{
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();
			_cfg.VistaPortNumber = VistALinkUnitTestHarness.TestData.VistaPortNumber;

			VerifyObject( _cfg, false, true, 0, null );
			Assert.AreEqual( VistALinkUnitTestHarness.TestData.VistaPortNumber, _cfg.VistaPortNumber );
		}

		[Test]
		public void VistaPortNumber_Fail_1()
		{
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();
			_cfg.VistaPortNumber = Common.StNullConvert.Int32Null;

			VerifyObject( _cfg, false, true, 1, BROKEN.VistaLinkConfig.VistaServerPortNumberNotSet );
			Assert.IsTrue( Common.StNullConvert.IsNull( _cfg.VistaPortNumber ) );
		}

		[Test]
		public void VistaPortNumber_Fail_2()
		{
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();
			_cfg.VistaPortNumber = InvalidVistaPortNumber;

			VerifyObject( _cfg, false, true, 1, BROKEN.VistaLinkConfig.VistaServerPortNumberOutOfRange );
			Assert.AreEqual( InvalidVistaPortNumber, _cfg.VistaPortNumber );
		}

		private void RunSaveTest()
		{
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();
		
			_cfg.VistaPortNumber = VistALinkUnitTestHarness.TestData.VistaPortNumber;
			_cfg.VistaIpAddress = VistALinkUnitTestHarness.TestData.VistaIpAddress;
		    _cfg.VbecsServiceIpAddress = IPAddress.Parse("IP_ADDRESS");
		    _cfg.VbecsServicePortNumber = PORT redacted;
			_cfg.Save( Common.UpdateFunction.UnitTests );

			_cfg = GetNonNullVerifiedConfig();

			VerifyObject( _cfg, false, false, 0, null );
			Assert.IsTrue( _cfg.VistaIpAddress.Equals( VistALinkUnitTestHarness.TestData.VistaIpAddress ) ); // Must do this way, so the IP values (not references) will be compared properly
			Assert.AreEqual( _cfg.VistaPortNumber, VistALinkUnitTestHarness.TestData.VistaPortNumber );
		}

		[Test]
		public void Save_Pass_1()
		{
			RunSaveTest();
		}

		[Test]
		public void Save_Pass_2()
		{
            UnitTestUtil.RunSQL("DELETE FROM " + TABLES.VistaLinkControl.TableName); 
			RunSaveTest();
		}

		[Test]
		[ExpectedException(typeof(BusinessObjectException))]
		public void Save_Fail()
		{
			VistALinkConfig _cfg = GetNonNullVerifiedConfig();
			_cfg.VistaPortNumber = InvalidVistaPortNumber;

			_cfg.Save( Common.UpdateFunction.UnitTests );
		}
	}
}
#endif
